within DisHeatLib.Boundary;
model SoilTemperature "Undisturbed soil temperature model"

  parameter DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp inputType=
      DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Constant
    "Model for ground temperature" annotation (choicesAllMatching=true);


  // Constant temperature
  parameter Modelica.SIunits.Temperature T_const(start=10.0 + 273.15) "Constant temperature"
    annotation(Dialog(enable= inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Constant));

  // Kusuda model
  import Modelica.Constants.pi;
  parameter Modelica.SIunits.ThermalDiffusivity alpha=5.96732e-7
    "Soil thermal diffusivity"
    annotation(Dialog(enable= inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Undisturbed));
  parameter Modelica.SIunits.Length z = 1 "Average depth of pipes"
    annotation(Dialog(enable= inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Undisturbed));
  parameter Modelica.SIunits.Temperature T_mean(start=8.65 + 273.15)
    "Average annual ground surface temperature"
    annotation(Dialog(enable= inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Undisturbed));
  parameter Modelica.SIunits.TemperatureDifference T_amp(start=17.146875)
    "Amplitude of annual ground surface temperature"
    annotation(Dialog(enable= inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Undisturbed));
  parameter Modelica.SIunits.Time t_min(start=353*24*60*60)
    "Time of the year presenting the minimum surface temperature"
    annotation(Dialog(enable= inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Undisturbed));

  Modelica.SIunits.Power Q_flow "Heat flow to/from soil (e.g., losses)";

protected
  Modelica.SIunits.ThermalDiffusivity alpha_day = alpha*86400
    "Soil thermal diffusivity per day";
  parameter Modelica.SIunits.Time startTime(fixed=false);
initial equation
  startTime=time;

public
  Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_b port annotation (
      Placement(transformation(extent={{-10,90},{10,110}}, rotation=0),
        iconTransformation(extent={{-10,90},{10,110}})));

equation
  if inputType==DisHeatLib.Boundary.BaseClasses.InputTypeSoilTemp.Undisturbed then
    port.T = T_mean - T_amp*exp(-z*sqrt(Modelica.Constants.pi/(365*alpha_day)))*
      cos(2*Modelica.Constants.pi/365*(time/86400 + startTime/86400 - t_min/86400 - z/2*sqrt(365/(
      Modelica.Constants.pi*alpha_day))));
  else
    port.T = T_const;
  end if;

  Q_flow = port.Q_flow;
  annotation (                  Documentation(info="<html>
<p>The model is based on the Kusuda equations, which describes the temperature on the ground as a function of the depth z and time. </p>
<p>To determine the temperature it is necessary to have information about the weather: mean annual temperature (T_mean) and annual temperature amplitude (T_amp) as well as information about the ground thermal diffusivity alpha. Furthermore the equation is tunned with a time shift parameter t_min which correponds to the day at which the ambient temperature has its lower value and a time shift parameter t_0 which has to be set to the day at which the simulation starts. </p>
<h4>Possible errors</h4>
<p>In case the temperature of the soil goes below zero, care must be taken to make sure that the water in the supply and return lines will not freeze. This can be omitted by, e.g., implementing a bypass, increase insulation of pipe, redimensioning the pipe (to increase mass flows), etc.</p>
<h4>References</h4>
<p>Earth temperature and thermal diffusivity at selected stations in the United States; Kusuda, T. and Achenbach P. R., 1965 </p>
</html>", revisions="<html>
<ul>
<li>Feburary 27, 2019, by Benedikt Leitner:<br>Implementation and added User&apos;s guide. </li>
</ul>
</html>"), Icon(graphics={
        Rectangle(
          lineColor={200,200,200},
          fillColor={248,248,248},
          fillPattern=FillPattern.HorizontalCylinder,
          extent={{-100.0,-100.0},{100.0,100.0}},
          radius=25.0),
                                 Text(
          extent={{-149,-99},{151,-139}},
          lineColor={0,0,255},
          fillPattern=FillPattern.HorizontalCylinder,
          fillColor={0,127,255},
          textString="%name"),
        Line(
          points={{-70,-62}},
          color={28,108,200},
          thickness=1),
        Rectangle(
          lineColor={128,128,128},
          extent={{-100.0,-100.0},{100.0,100.0}},
          radius=25.0),
        Line(
          points={{-100,60},{-52,60},{100,60}},
          color={162,29,33},
          thickness=1),
        Ellipse(
          extent={{-78,2},{-74,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,-8},{-64,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-58,2},{-54,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,12},{-64,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-58,-18},{-54,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,-28},{-44,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,-18},{-34,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,-8},{-44,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-58,22},{-54,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,12},{-44,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,22},{-34,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,32},{-44,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,2},{-34,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,-8},{-24,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,2},{-14,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,12},{-24,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,-38},{-34,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,-48},{-24,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,-38},{-14,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,-28},{-24,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,-18},{-14,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,-28},{-4,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,-18},{6,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,-8},{-4,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,42},{-34,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,32},{-24,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,42},{-14,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,52},{-24,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,22},{-14,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,12},{-4,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,22},{6,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,32},{-4,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-10,52},{-6,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,42},{6,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,32},{16,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,42},{26,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,52},{16,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,2},{6,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,-8},{16,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,2},{26,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,12},{16,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,22},{26,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,12},{36,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,22},{46,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,32},{36,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,-58},{-14,-62}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,-68},{-4,-72}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,-58},{6,-62}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,-48},{-4,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,-78},{6,-82}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,-88},{16,-92}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,-78},{26,-82}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,-68},{16,-72}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{2,-38},{6,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{10,-48},{14,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,-38},{26,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{12,-28},{16,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,-58},{26,-62}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,-68},{36,-72}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,-58},{46,-62}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,-48},{36,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-8,-88},{-4,-92}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,-70},{-24,-74}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{72,-88},{76,-92}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,-88},{36,-92}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,-78},{46,-82}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{52,-88},{56,-92}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{62,-78},{66,-82}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{52,-68},{56,-72}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{22,-18},{26,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,-28},{36,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,-18},{46,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{32,-8},{36,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,-38},{46,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{52,-48},{56,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{62,-38},{66,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{52,-28},{56,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,2},{46,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{50,-8},{54,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{62,2},{66,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{52,12},{56,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{62,-18},{66,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{72,-28},{76,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{82,-18},{86,-22}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{72,-8},{76,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{62,-58},{66,-62}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{72,-68},{76,-72}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{82,-58},{86,-62}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{72,-48},{76,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{82,-38},{86,-42}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{92,-48},{96,-52}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{92,-68},{96,-72}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{92,-28},{96,-32}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,-80},{-34,-84}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,-90},{-44,-94}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-98,-40},{-94,-44}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-28,-90},{-24,-94}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-98,-60},{-94,-64}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,-70},{-84,-74}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-78,-60},{-74,-64}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,-50},{-84,-54}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-98,-20},{-94,-24}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,-30},{-84,-34}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-78,-20},{-74,-24}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,-10},{-84,-14}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-78,-40},{-74,-44}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,-50},{-64,-54}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-58,-40},{-54,-44}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,-30},{-64,-34}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-78,-80},{-74,-84}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,-90},{-64,-94}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-58,-80},{-54,-84}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,-70},{-64,-74}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-58,-60},{-54,-64}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,-70},{-44,-74}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-38,-60},{-34,-64}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,-50},{-44,-54}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{34,52},{38,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{42,42},{46,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{54,52},{58,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{54,32},{58,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{62,22},{66,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{74,32},{78,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{64,42},{68,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{74,52},{78,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{84,42},{88,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{94,52},{98,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{74,12},{78,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{82,2},{86,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{94,12},{98,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{84,22},{88,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{94,32},{98,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{82,-78},{86,-82}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{92,-8},{96,-12}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-18,-80},{-14,-84}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-98,2},{-94,-2}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,12},{-84,8}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-78,22},{-74,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,32},{-84,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-98,22},{-94,18}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-98,42},{-94,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-88,52},{-84,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-78,42},{-74,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-66,52},{-62,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-68,32},{-64,28}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-56,42},{-52,38}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None),
        Ellipse(
          extent={{-48,52},{-44,48}},
          lineThickness=1,
          fillColor={162,29,33},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None)}));
end SoilTemperature;
